/* 
* oneWireNet.h
*
* Created: 16/06/2019 23:32:35
* Author: MAN's
*/

/*
 oneWireNet.h - Base class that provides some 1-wire prototypes
 
 This library is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 Lesser General Public License for more details.
 
 */



#ifndef ONEWIRENET_H_
#define ONEWIRENET_H_



#ifndef _AVR_INTERRUPT_H_
#include "avr/interrupt.h"
#endif

#include "ow_delays.h"



class OneWireNet
{

public:
#define _RomcrcIdx              0x07
#define _SPadcrcIdx             0x08

#define _DS_READROM             0x33
#define _DS_MATCHROM            0x55
#define _DS_SEARCHROM           0xF0
#define _DS_ALASEARCH           0xEC
#define _DS_SKIPROM             0xCC
#define _DS_CONVTEMP            0x44
#define _DS_COPYSCRATCHPAD      0x48
#define _DS_READSUPPLY          0xB4
#define _DS_RECALLEEPROM        0xB8
#define _DS_READSCRATCHPAD      0xBE
#define _DS_WRITESCRATCHPAD     0x4E
#define _DS_ACTIVEONLYSEARCH   0xEC

#define _DS_RES9BIT             31
#define _DS_RES10BIT            63
#define _DS_RES11BIT            95
#define _DS_RES12BIT            127

#define Low2Hi    0
#define Hi2Low    1

#define byte uint8_t



//variables
byte endSearch;
byte LastFamilyDiscrepancy;


typedef enum {
    badReset = -1,      //No reset for device was performed
    noDevice = -2,      //Conflict between devices on 1-wire bus.  
    goodSearch = 127    //search performed successfully. 

} s_flags;


/*
    Calculates the CRC of device array ROM and Scratchpad.  
    Param: pCRC is the pointer of array.
    Param: pCRCidx is the lenght of array pCRC minus one.
    return true if CRC is OK.
*/
bool ChkCRC(byte *pCRC, const byte pCRCidx);

/*  detects if a pull up resistor is connected to the 1-wire pin
    return true if a resistor is connected or false otherwise.
*/
bool PUPdetect();

/*
    Perform a device reset
    return true if success.
*/
bool Reset_Dev();

/*
    Search by all devices attached to the 1-wire bus
    Param: ROMID is ROM array that holds all ROM data
    Param: pCmd is ROM command constant to perform the search.
    Return: true if search succeed.   
*/
bool Search(byte *romID, byte pCmd);

/*
    Return false if device is used as parasite mode
    or true if in powered mode.
    Param: ROMID is ROM array for that device.
*/
bool Pwr_Type(byte *romID);

/*
    Selects the specific device on the bus 1-wire.
    Param: romID is the specific ROM array for that
    device on the bus.
    return true if success otherwise false.
*/

bool Select_Dev(byte *romID);

/*
    Search devices on 1-wire bus
    Param: romID is ROM array that holds the ROM info of device.
    Param: pCmd is the constant ROM command for desired task.
    return true if success.
*/

char Search_Dev(byte* romID, byte pCmd);

/*
    Detects device(s) and fill a pre-defined buffer with devices
    attached to the bus 1-wire. This function can driven up to
    127 devices. The MSBit will hold the CRC error if occurred, this
    position is set to 1 during the search, the amounth of devices
    is get by masking this value with 0x7F or any value higher 127 
    will have an CRC error.
    
    Param : romID is the ROM array[n][8] to be filled.
    return: the amounth of devices found on the bus 1-wire plus
            CRC error if one was detecteded.
            
*/
byte DeviceFillBuffer(byte *romID);

/*
    Counts the amounts of devices attached to 1-wire bus
    return the number of devices found in the bus.
*/
byte Count_Device();

/*
    Prepares, copy the scratch pad to a internal buffer before
    it is read by read_ScratchPad function.
    Param: romID is the specific ROM of device that we want to read
    the scratch pad.
*/
void Copy_SPad(byte *romID);

/*
    Perform an internal device convert temperature
    this function looks for powered device mode
    to apply the delay conversion. if the device
    is in powered mode the time wait is hardware 
    detect if parasite mode a fixed delay is used.
    Param: romID is the array of device to intend to convert temp.
    
*/
void ConvTemp(byte *romID);

/*
    Writes to internal scratch pad with 9 bits  res.
    Param: romID is the specific ROM of device that we want to
    write the set point temperature for alarms.
    Param: tHiB is the hi limit temperature for alarm (byte range).
    Param: tLoB is the lo limit temperature for alarm (byte range).
*/
void WriteSpad9(byte *romID, byte tHiB, byte tLoB);

/*
    Writes to internal scratch pad with 12 bits res.
    Param: romID is the specific ROM array of device that we want to
    write the set point temperature for alarms.
    
    Param: tHiB is the hi limit temperature for alarm (byte range).
    Param: tLoB is the lo limit temperature for alarm (byte range).
    Param: ConfByte is the specific configuration see data sheet of
    device.
*/
void WriteSpad12(byte *romID, byte tHiB, byte tLoB, byte ConfByte);

/*
    Perform recall E2PROM for a specific device
    Param: romID is the specific ROM array of device.
*/
void RcallE2Prom(byte *romID);

/*
    Perform a read ROM and fill an array with its values.
    Param: romID a pointer to array that will holds the ROM data.
*/
void Read_Rom(byte *romID);

/*
    Perform a read scratch pad and fill an array with its values.
    Param: pSpad_Array a pointer to array that will holds the scratch pad data.
*/
void ReadSPad(byte *pSpad_Array);

/*
    Perform a command to any device.
    Param: pCmdVal is one of constants that takes a command.
*/
void Issue_Cmd(byte pCmdVal);

void Issue_Data(const byte data);

byte GetData();


/*
    Initializes a 1-wire pin to holds the net 1-wire.
    Param: pin is constant that point to pin to be used.
*/
OneWireNet(volatile uint8_t* port, const byte pin);

OneWireNet(byte pin);


/*
            Only used in Parasite Mode for real use.

      Determines what pin will be used for control the power in parasite mode.
      Parameter port: is a pointer to the port that will control the power pin for 
                      parasite mode poered.
      Parameter pin:  is the pin control associated to the port to fire a strong pull-up.
      Parameter type: control the polarIty that activates the STRONG PULL-UP (low2high or Hi2Low)
      Parameter Int:  if interrupt is active we could inform the procedure to drive these interrupt
                      when long delays is in course.
*/
void strong_pull_Up(volatile uint8_t* port, const byte pin, byte pol_type, byte Int);




private:

    #define DQ_READ(reg,mask)           (((*(reg)) & (mask)) ? 1 : 0)                               //all reg is the register base DDRx passed as pointer.
    #define DQ_HI_Z(reg,mask)           ((*(reg)) &= ~(mask));((*(reg+1)) &= ~(mask));asm("NOP")    //DDRx = 0; PORTx = 0
    #define DQ_LOW_Z(reg,mask)          ((*(reg)) |= (mask));((*(reg+1)) &= ~(mask));asm("NOP")     //DDRx = 1; PORTx = 0
    #define DQ_OUTPUT_HI(reg,mask)      ((*(reg)) |= (mask));((*(reg+1)) |= (mask));asm("NOP")      //DDRx = 1; PORTx = 1


    typedef union {
        unsigned char   value;
        struct {
            unsigned    Bit0:1;
            unsigned    Bit1:1;
            unsigned    Bit2:1;
            unsigned    Bit3:1;
            unsigned    Bit4:1;
            unsigned    Bit5:1;
            unsigned    Bit6:1;
            unsigned    Bit7:1;
        }bits;
    }crc_s;


    volatile byte* PWrite;
    volatile byte* PRead;
    volatile byte* PDir;

    byte bitmask;

    byte ReadBit();
    void WriteBit(byte V);
    void Write_Byte(byte bytecmd);
    byte Read_Byte();
    void Ow_WaitBusy(uint16_t dly);
    void initOwire(volatile uint16_t port, const byte pin);

};

#endif
