/*
* oneWireNet.cpp
*
* Created: 16/06/2019 23:32:35
* Author: MAN's
*/

/*
 oneWireNet.cpp - Base class that provides 1-wire functions
 
 This library is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 Lesser General Public License for more details.

 */


#include "oneWireNet.h"



void OneWireNet::initOwire(uint16_t port, const byte pin)
{
    MCUCR |= (1 << PUD);
    PWrite = port;
    PDir  = PWrite - 1;
    PRead = PWrite - 2;
    bitmask = (1 << (pin));
    DQ_HI_Z(PDir,bitmask);
}

// default constructor
OneWireNet::OneWireNet(volatile  uint8_t* port, const byte pin)
{
    initOwire(port,pin);   
}

/*
    The analogs pins are not used and not maped.
    Map for Atmega 328p and its derivates.
*/

OneWireNet::OneWireNet(byte pin)
{
    uint16_t port;
    switch(pin)
    {
        case    0:
        case    1:
        case    2:
        case    3:
        case    4:
        case    5:
        case    6:
        case    7:
            port = &PORTD;
            break;
        case    8:
        case    9:
        case    10:
        case    11:
        case    12:
        case    13:
            port = &PORTB;
            pin -= 8;
            break;
        default :
            break;
    }
    initOwire(port,pin);
}


bool OneWireNet::ChkCRC(byte *pCRC,byte pCRCidx){ 
    byte index, BitIndex;
    crc_s CRC;
    crc_s lResult;
    crc_s ByteData;
    CRC.value = 0;
    lResult.value = 0;
    index = 0;
    while  (index <= pCRCidx){
        ByteData.value = pCRC[index];                                       
        //'Do checksum...
        for (BitIndex = 0;BitIndex<8;BitIndex++){                           
            lResult.bits.Bit0 = (CRC.bits.Bit0 ^ ByteData.bits.Bit0);       
            ByteData.value  = ByteData.value >> 1;                          
            if (lResult.value == 0x01) CRC.value ^= 0x18;                       
            CRC.value = CRC.value >> 1;                                     
            CRC.bits.Bit7 = lResult.bits.Bit0;                          
        }
        index++;
    }
    lResult.value = (CRC.value == false);
    return lResult.value;
}


void OneWireNet::WriteBit(byte V){
    if (V == 0){
        DQ_LOW_Z(PDir,bitmask);
        Delay_ow_60us();
        DQ_HI_Z(PDir,bitmask);
        Delay_ow_1us();
    }else{
        DQ_LOW_Z(PDir,bitmask);
        Delay_ow_1us();
        DQ_HI_Z(PDir,bitmask);
        Delay_ow_60us();
    }
}

byte OneWireNet::ReadBit()
{
    byte res = 0;
    DQ_LOW_Z(PDir,bitmask);
    Delay_ow_1us();
    DQ_HI_Z(PDir,bitmask);
    Delay_ow_15us();
    res = DQ_READ(PRead,bitmask);
    Delay_ow_46us();
    return res;
}

byte OneWireNet::Read_Byte(){
    byte j;
    byte result = 0;
    for (j = 0;j<8; j++){
        if (j > 0) result >>= 1;
        if (OneWireNet::ReadBit() == 1) result |= 0x80;   
    }
    return result;
}

void OneWireNet::Write_Byte(byte bytecmd){
    byte j = 0;
    for (;j<8;j++){
        if ((bytecmd & 0x01) == 0x01){
            OneWireNet::WriteBit(1);
        }else{
            OneWireNet::WriteBit(0);
        }
        bytecmd >>= 1;
    }
}

bool OneWireNet::PUPdetect(){
    DQ_HI_Z(PDir,bitmask);
    Delay_ow_480us();
    return (DQ_READ(PRead,bitmask) > 0) ? true : false;
}

bool OneWireNet::Reset_Dev(){
    bool result =false;
    DQ_LOW_Z(PDir,bitmask);
    Delay_ow_480us();
    Delay_ow_75us();
    DQ_HI_Z(PDir,bitmask);
    Delay_ow_75us();
    if (!DQ_READ(PRead,bitmask)){
        Delay_ow_405us();
        result = true;
    }
    return result;
}


bool OneWireNet::Select_Dev(byte *romID){
    byte K;
    if (!OneWireNet::Reset_Dev()) return false;
        OneWireNet::Write_Byte(_DS_MATCHROM);
    for (K = 0;K<8;K++){
        OneWireNet::Write_Byte(romID[K]);
    }
    return true;
}

bool OneWireNet::Pwr_Type(byte *romID){
    bool ptype = false;
    if (OneWireNet::Select_Dev(romID)){
        OneWireNet::Write_Byte(_DS_READSUPPLY);
        ptype = (OneWireNet::Read_Byte() > 0) ? true : false;
        OneWireNet::Reset_Dev();
    }
    return ptype;
}

void OneWireNet::Copy_SPad(byte *romID){
    byte pwr_type = 0;
    OneWireNet::Select_Dev(romID);
    OneWireNet::Write_Byte(_DS_READSUPPLY);
    pwr_type = (OneWireNet::Read_Byte() > 0) ? 1 : 0;       
    OneWireNet::Select_Dev(romID);
    OneWireNet::Write_Byte(_DS_COPYSCRATCHPAD);
    if (pwr_type == 0) {
        DQ_HI_Z(PDir,bitmask);
        Ow_WaitBusy(SCRATCHPAD_DLY);
        DQ_HI_Z(PDir,bitmask);
    }else{
        while(!OneWireNet::ReadBit());
    }
    OneWireNet::Reset_Dev();
}

void OneWireNet::ConvTemp(byte *romID){
    byte pSuply = 0;
    OneWireNet::Select_Dev(romID);
    OneWireNet::Write_Byte(_DS_READSUPPLY);
    pSuply = (OneWireNet::Read_Byte() > 0) ? 1 : 0;
    OneWireNet::Select_Dev(romID);
    Write_Byte(_DS_CONVTEMP);
    if (pSuply == 0){
        DQ_HI_Z(PDir,bitmask);
        Ow_WaitBusy(CONVTEMP_DLY);
        DQ_HI_Z(PDir,bitmask);
    }else{
        while(!OneWireNet::ReadBit());
    }
    OneWireNet::Reset_Dev();
}


void OneWireNet::WriteSpad9(byte *romID, byte tHiB, byte tLoB){
    OneWireNet::Select_Dev(romID);
    OneWireNet::Write_Byte(_DS_WRITESCRATCHPAD);
    OneWireNet::Write_Byte(tHiB);
    OneWireNet::Write_Byte(tLoB);
    OneWireNet::Reset_Dev();
}


void OneWireNet::WriteSpad12(byte *romID, byte tHiB, byte tLoB, byte ConfByte){
    OneWireNet::Select_Dev(romID);
    OneWireNet::Write_Byte(_DS_WRITESCRATCHPAD);
    OneWireNet::Write_Byte(tHiB);
    OneWireNet::Write_Byte(tLoB);
    OneWireNet::Write_Byte(ConfByte);
    OneWireNet::Reset_Dev();
}

void OneWireNet::RcallE2Prom(byte *romID){
    OneWireNet::Select_Dev(romID);
    OneWireNet::Write_Byte(_DS_RECALLEEPROM );
    while(OneWireNet::ReadBit() == 0);
    OneWireNet::Reset_Dev();
}


void OneWireNet::Read_Rom(byte *romID){
    byte P = 0;
    OneWireNet::Write_Byte(_DS_READROM);
    do{
        *romID++ = OneWireNet::Read_Byte();
    }while(P++ < 8);
}

void OneWireNet::ReadSPad(byte *pSpad_Array ){
    byte P = 0;
    OneWireNet::Write_Byte(_DS_READSCRATCHPAD);
    do{
        *pSpad_Array++ = OneWireNet::Read_Byte();
    }while(P++ < 9);
}


char OneWireNet::Search_Dev(byte* romID, byte pCmd){
    byte Id_Bit_Number,
    Last_Zero,
    Rom_Byte_Number,
    Id_Bit,
    Cmp_Id_Bit,
    Rom_Byte_Mask,
    Search_Direction;
    Id_Bit_Number = 1;
    Last_Zero  = 0;
    Rom_Byte_Number = 0;
    Rom_Byte_Mask = 1;
    if (!OneWireNet::Reset_Dev()) return s_flags::badReset;
    OneWireNet::Write_Byte(pCmd);
    do{
        Id_Bit = 0;
        Cmp_Id_Bit = 0;
        Id_Bit = OneWireNet::ReadBit();
        Cmp_Id_Bit = OneWireNet::ReadBit();

        //Check for no device in the ow bus
        if (Id_Bit & Cmp_Id_Bit){
            return s_flags::noDevice;
        }else{
            //All device Coupled have 0 or 1
            if (Id_Bit ^ Cmp_Id_Bit){
                Search_Direction = Id_Bit;
            }else{
                if (Id_Bit_Number < endSearch){
                    Search_Direction = ((*romID & Rom_Byte_Mask) > 0);
                }else{
                    Search_Direction = (Id_Bit_Number == endSearch);
                }
                if (Search_Direction == 0) Last_Zero = Id_Bit_Number;
                if (Last_Zero < 9) LastFamilyDiscrepancy = Last_Zero;
            }
            OneWireNet::WriteBit(Search_Direction);
            if (Search_Direction > 0){
                *romID |= Rom_Byte_Mask;
            }else{
                *romID &= ~Rom_Byte_Mask;
            }
            Id_Bit_Number++;
            Rom_Byte_Mask = (Rom_Byte_Mask << 1);
            if (Rom_Byte_Mask == 0) {
                Rom_Byte_Number++;
                romID++;
                Rom_Byte_Mask = 1;
            }
        }
    }while(Rom_Byte_Number < 8);
    endSearch = Last_Zero;
    Rom_Byte_Mask = 0;
    return s_flags::goodSearch;
}


byte OneWireNet::DeviceFillBuffer(byte *romID){
    byte RomTmp[8];
    byte count;
    byte Result = 0;
    do
    {
        count = 0;
        if (OneWireNet::Search_Dev(RomTmp,_DS_SEARCHROM) == goodSearch)
        {
            if (OneWireNet::ChkCRC(RomTmp,_RomcrcIdx))
            {
                do
                {
                    romID[count] = RomTmp[count];
                    count++;
                }while(count < 8);
                Result++;
                romID = romID + count;
            }
            else
            {
                Result |= 0x80;
            }
        }
    }while (endSearch > 0);
    return Result;
}


byte OneWireNet::Count_Device(){
    byte DumyCount[8];
    byte Result = 0;
    do
    {
        if (OneWireNet::Search_Dev(DumyCount, _DS_SEARCHROM)  == goodSearch) { Result++; }
    }while(endSearch > 0);
    return Result;
}

void OneWireNet::Issue_Cmd(byte pCmdVal){
    OneWireNet::Write_Byte(pCmdVal);
}

void OneWireNet::Issue_Data(const byte data)
{
    OneWireNet::Write_Byte(data);
}

byte OneWireNet::GetData()
{
    return OneWireNet::Read_Byte();
}

volatile uint8_t* pu_port;
volatile uint8_t* pu_dir;
bool Interrupt;
uint8_t pu_pin;
bool pull_up_In_Use;
uint8_t polarity = Low2Hi;

void OneWireNet::strong_pull_Up(volatile uint8_t* port, const byte pin, byte pol_type, byte Int)
{
    if ( pin < 0) return;
    pu_port = port;
    pu_dir = port - 1;
    *pu_dir |= (1 << pin);
    Interrupt = (Int > 0) ? 1 : 0;
    pull_up_In_Use = false;
    pol_type = (pol_type >  0) ? 1 : 0;
    pu_pin = (uint8_t)pin;
    pull_up_In_Use = true;
    polarity = pol_type;
}

void OneWireNet::Ow_WaitBusy(uint16_t dly)
{
    if (pull_up_In_Use && polarity == Low2Hi)
    {
        *pu_port |= (1 << pu_pin);
    }
    else if (pull_up_In_Use && polarity == Hi2Low)
    {
        *pu_port &= ~(1 << pu_pin);
    }
    
    /*
        we can re enable interrupts if necessary during 
        this time
    */
    if (Interrupt)  sei();
    
    while(dly-- > 0)
    {
        _delay_us(1000);
    }

    if (pull_up_In_Use && polarity == Low2Hi){
        *pu_port &= ~(1 << pu_pin);
    }
    else if (pull_up_In_Use && polarity == Hi2Low)
    {
        *pu_port |= (1 << pu_pin);
    }
    
    /*  but we need to disable to continue,
        1-wire is a bit time slot and is software
        timed if interrupt is in use error can occur.
    */
    
    if (Interrupt) cli();
}

